//  AppRunSettings Class
//
//  Defines the command line breakout for the Instrumentation Package Analyser application
//
//  Version:  1.0.0
//  Build:    01
//
//   Copyright 2011 Hadleigh Marshall Netherlands b.v.

//  Application includes
#include "AppRunSettings.h"

//  AppRunSettings Default Constructor
AppRunSettings::AppRunSettings(void)
{
	//  Initialise the default settings
	SetDefaults();
}

//  AppRunSettings Normal Constructor
AppRunSettings::AppRunSettings(int argc, char * argv[])
{
	//  Initialise the default settings
	SetDefaults();
	//  Parse and validate command line parameters
	IsValid = ValidateParameters(argc, argv);
}

//  AppRunSettings Destructor
AppRunSettings::~AppRunSettings(void)
{

}

//  SetDefaults
//
//  This function will set the default values for the Application settings
//
//  Parameters
//
//  Returns
//
//  BOOL  -				TRUE if the defaults were set
//						FALSE if the defaults were not set
//
//  API Use Notes:
//
//
BOOL AppRunSettings::SetDefaults(void)
{
	//  Set the default values for all settings
	IsValid = FALSE;							//  Default to parameter set is invalid
	AllowExecution = FALSE;						//  Allow execution to continue
	szFileName[0] = '\0';						//  Clear the filename string
	Outputs = 0;								//  No output options are selected
	return TRUE;
}

//  ValidateParameters
//
//  This function will validate and the application command line parameters and set the appropriate
//  settings for the program execution.
//
//  Parameters
//
//  Returns
//
//  BOOL  -				TRUE if the parameters were valid
//						FALSE if the parameters were invalid
//
//  API Use Notes:
//
//
BOOL AppRunSettings::ValidateParameters(int argc, char* argv[])
{
	BOOL		bFlagIsValid;											//  Indicator if a parameter flag is valid
	int			iIndex;													//  Generic Index
	char		szFlag[MAX_FLAG + 1];									//  Local storage for flag value
	char		szMsg[MAX_MSG + 1];										//  Generic message buffer

	//  Validate and capture the runtime parameters
	AllowExecution = TRUE;

	//  See if we are just being asked to display usage
	if (argc == 1)
	{
		ShowUsage();
		AllowExecution = FALSE;
		return TRUE;
	}

	strcpy_s(szFileName, MAX_FILE, argv[1]);
	if (szFileName[0] == '-')
	{
		if (szFileName[1] == '?')
		{
			ShowUsage();
			AllowExecution = FALSE;
			return TRUE;
		}

		//  Error - Filename must be specified
		std::cout << MSG_IPA0001E << std::endl;
		ShowUsage();
		return FALSE;
	}

	if (argc > 2)
	{
		for (iIndex = 2; iIndex < argc; iIndex++)
		{
			bFlagIsValid = FALSE;
			strcpy_s(szFlag, MAX_FLAG, argv[iIndex]);
			if (memcmp(szFlag, "-", 1) == 0)
			{
				//  Parameter is a switch
				if (memcmp(szFlag + 1, "?", 1) == 0)
				{
					//  Show Usage - do not run the application
					ShowUsage();
					AllowExecution = FALSE;
					return TRUE;
				}

				if ((memcmp(szFlag + 1, "D", 1) == 0) || (memcmp(szFlag + 1, "d", 1) == 0))
				{
					Outputs = Outputs | OUTPUT_DUMP;
					bFlagIsValid = TRUE;
				}

				if ((memcmp(szFlag + 1, "S", 1) == 0) || (memcmp(szFlag + 1, "s", 1) == 0))
				{
					Outputs = Outputs | OUTPUT_STATS;
					bFlagIsValid = TRUE;
				}

				if ((memcmp(szFlag + 1, "C", 1) == 0) || (memcmp(szFlag + 1, "c", 1) == 0))
				{
					Outputs = Outputs | OUTPUT_CHART;
					bFlagIsValid = TRUE;
				}

				if (!bFlagIsValid)
				{
					//  Invalid switch
					sprintf_s(szMsg, MAX_MSG, MSG_IPA0002E, szFlag);
					std::cout << szMsg << std::endl;
				}
			}
		}
	}

	//  Check that some output was requested - if none then default to -S statistics report
	if (Outputs == 0) Outputs = OUTPUT_STATS;

	return TRUE;
}

//  ShowUsage
//
//  This function will show the application command line usage pattern.
//
//  Parameters
//
//  Returns
//
//  API Use Notes:
//
//
void AppRunSettings::ShowUsage(void)
{
	//  Show program usage

	std::cout << MSG_USAGE1 << std::endl;
	std::cout << MSG_USAGE2 << std::endl;
	std::cout << MSG_USAGE3 << std::endl;
	std::cout << MSG_USAGE4 << std::endl;
	std::cout << MSG_USAGE5 << std::endl;
	std::cout << MSG_USAGE6 << std::endl;
	std::cout << MSG_USAGE7 << std::endl;
	return;
}

