// DXIPA.cpp : Defines the entry point for the DX Instrumentation Package Analyser application.
//
//*******************************************************************************************************************
//*																													*
//*   File:       DXIPA.cpp																							*
//*   Suite:      Domino Tools																						*
//*   Version:    1.0.1	  Build:  03																				*
//*   Author:     Ian Tree/HMNL																						*
//*																													*
//*   Copyright 2011 - 2012 Hadleigh Marshall Netherlands b.v.														*
//*******************************************************************************************************************
//*	DXIPA																											*
//*																													*
//*	This application will print and analyse the contents of a DX Instrumentation Package Recording (.ipr) file.   	*
//* all output is written to STDOUT.																				*
//*																													*
//*	USAGE:																											*
//*																													*
//* DXIPA <filename> [-D][-S][-C]																					*
//*																													*
//*	<filename>				- Name of the Instrumentation Package Analyser file to analyse							*
//*	-D						- Dump (print) the contents																*
//*	-S						- Print a statistical analysis of the contents											*
//* -C						- Chart a statistical analysis of the contents											*
//*																													*
//*	NOTES:																											*
//*																													*
//*	If no output options are specified then statistical anaysis (-S) is performed.									*
//*																													*
//*																													*
//*******************************************************************************************************************
//*																													*
//*   History:																										*
//*																													*
//*	1.0.0 - 07/12/2011   -  Initial Version																			*
//* 1.0.1 - 11/01/2012	 -  Arithmetic corrections																	*
//*																													*
//*******************************************************************************************************************/

//  Application includes
#include	"DXIPA.h"

int main(int argc, char* argv[])
{
	AppRunSettings			*arsLocal;							//  Applictaion Run Settings Object

	//  Parse the application run settings
	arsLocal = new AppRunSettings(argc, argv);
	if (!arsLocal->AllowExecution)								//  Exit silently if just showing usage
	{
		delete arsLocal;
		return APPRC_NOERROR;
	}

	if (!arsLocal->IsValid)										//  Abort if validation failed
	{
		delete arsLocal;
		std::cout << MSG_IPA0050E << std::endl;
		return APPRC_FATAL;
	}

	//  Annotate the dump
	std::cout << APP_TITLE << " (" << APP_NAME << ") Version: " << APP_VERSION << " analysing: " << arsLocal->szFileName << std::endl;
	std::cout << std::endl;
	std::cout << std::endl;

	//  Analyse the requested file
	AnalyseThisFile(arsLocal->szFileName, arsLocal->Outputs);

	//  Terminate the application
	std::cout << std::endl;
	std::cout << std::endl;
	std::cout << APP_TITLE << " (" << APP_NAME << ") Version: " << APP_VERSION << " analysis completed." << std::endl;
	return APPRC_NOERROR;
}

//  AnalyseThisFile
//
//  This function will execute the analysis of the passed instrumentation package recording file.
//
//  Parameters
//
//  1		-	char *			-  pointer to the null terminated file name to be analysed
//  2		-	int				-  output options required for this analysis
//
//  Returns
//
//  API Use Notes:
//
//
void AnalyseThisFile(char *szFileName, int iOutputs)
{
#ifdef WIN32
	HANDLE				hfSrc;									//  Handle of the source file
	DWORD				dwWinError;								//  Last error code
#endif
#ifdef UNIX
	int					hfSrc;									//  Descriptor of the source file
	off_t				offSrc;											//  Source Offset
	off_t				offResult;										//  Resulting offset from lseek
#endif
	char				szMsg[MAX_MSG + 1];						//  Generic message buffer

#ifdef WIN32
	//  Open the source file
	hfSrc = CreateFile(szFileName, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hfSrc == (HANDLE) HFILE_ERROR)
	{
		dwWinError = GetLastError();
		sprintf_s(szMsg, MAX_MSG, MSG_IPA0100E, szFileName);
		std::cout << szMsg << std::endl;
		GetWINAPIError(dwWinError, szMsg);
		std::cout << szMsg << std::endl;
		return;
	}
#endif

#ifdef UNIX
	hfSrc = open(szFileName, O_RDONLY | O_LARGEFILE);
	if (hfSrc == -1)
	{
		sprintf_s(szMsg, MAX_MSG, MSG_IPA0100E, szFileName);
		std::cout << szMsg << std::endl;
		GetUNIXAPIError(errno, szMsg);
		std::cout << szMsg << std::endl;
		return;
	}
	offSrc = 0;
	offResult = lseek(hfSrc, offSrc, SEEK_SET);
#endif

	//  Analyse all of the records in the passed file
	AnalyseTheseRecords(hfSrc, iOutputs);

	//  Close the file
#ifdef WIN32
	CloseHandle(hfSrc);
#endif

#ifdef UNIX
	close(hfSrc);
#endif
	return;
}

#ifdef WIN32
//  AnalyseTheseRecords
//
//  This function will execute the analysis of all of the IP records in the passed file
//
//  Parameters
//
//  1		-	HANDLE			-  handle of the file containing the record stream to be analysed
//  2		-	int				-  output options required for this analysis
//
//  Returns
//
//  API Use Notes:
//
//
void AnalyseTheseRecords(HANDLE hfIPR, int iOutputs)
#endif

#ifdef UNIX
//  AnalyseTheseRecords
//
//  This function will execute the analysis of all of the IP records in the passed file
//
//  Parameters
//
//  1		-	int				-  descriptor of the file containing the record stream to be analysed
//  2		-	int				-  output options required for this analysis
//
//  Returns
//
//  API Use Notes:
//
//
void AnalyseTheseRecords(int hfIPR, int iOutputs)
#endif

{
	PCRecRC				pcrCurrent;										//  Current Instrumentation Package Record
	BOOL				bKeepReading;									//  Control of the read loop
	DWORD				dwRecs;											//  Count of records read
	DWORD				dwBytesRead;									//  Count of bytes read
	DWORD				dwPosLo;										//  File position (Low)
#ifdef WIN32
	DWORD				dwPosHi = 0;									//  File position (High)
	DWORD				dwWinError;										//  Windows Error Code
#endif
#ifdef UNIX
	off_t				offSrc;											//  Source Offset
	off_t				offResult;										//  Resulting offset from lseek
	ssize_t				stRead;											//  Size read/written
#endif
	AutoScaleFrequencyAnalyser	*asfaDwell;								//  Analyser for the Dwell Time
	AutoScaleFrequencyAnalyser	*asfaMill;								//  Analyser for the Mill Time
	char				szMsg[MAX_MSG + 1];								//  Generic message buffer

	bKeepReading = TRUE;
	dwRecs = 0;

	//  Create frequency analysers for the dwell and mill times
	asfaDwell = new AutoScaleFrequencyAnalyser(100, 10, 5);
	asfaMill = new AutoScaleFrequencyAnalyser(100, 10, 5);

	//  Process Each Record in turn
	while (bKeepReading)
	{
		//  Read the next record - start with the record prefix
#ifdef WIN32
		if (!ReadFile(hfIPR, &pcrCurrent, sizeof(int) * 2, &dwBytesRead, NULL))
#endif
#ifdef UNIX
		stRead = read(hfIPR, &pcrCurrent, sizeof(int) * 2);
		if (stRead == -1)
#endif
		{
			//  EOF
			std::cout << MSG_IPA0102I << std::endl;
			bKeepReading = FALSE;
		}
		else
		{
#ifdef UNIX
			dwBytesRead = stRead;
#endif
			if (dwBytesRead != (sizeof(int) * 2))
			{
				//  EOF
				std::cout << MSG_IPA0102I << std::endl;
				bKeepReading = FALSE;
			}
			else
			{
				//  Check the type and the length
				if (pcrCurrent.PCBase.Type != DXIPR_RQE_COMPLETED)
				{
					//  Skip the record
					sprintf_s(szMsg, MAX_MSG, MSG_IPA0103I, dwRecs + 1, pcrCurrent.PCBase.Type, pcrCurrent.PCBase.Size); 
					std::cout << szMsg << std::endl;
					dwPosLo = pcrCurrent.PCBase.Size - (sizeof(int) * 2);
#ifdef WIN32
					if(SetFilePointer(hfIPR, dwPosLo, (PLONG) &dwPosHi, FILE_CURRENT) == MAXWORD)
					{
						dwWinError = GetLastError();
						sprintf_s(szMsg, MAX_MSG, MSG_IPA0104E, dwPosLo);
						std::cout << szMsg << std::endl;
						GetWINAPIError(dwWinError, szMsg);
						std::cout << szMsg << std::endl;
						bKeepReading = FALSE;
					}
#endif
#ifdef UNIX
					offSrc = dwPosLo;
					offResult = lseek(hfIPR, offSrc, SEEK_CUR);
					if (offResult == -1)
					{
						sprintf_s(szMsg, MAX_MSG, MSG_IPA0104E, dwPosLo);
						std::cout << szMsg << std::endl;
						GetUNIXAPIError(errno, szMsg);
						std::cout << szMsg << std::endl;
						bKeepReading = FALSE;
					}
#endif
				}
				else
				{
					//  Check the length - skip the record if it does not agree
					if (pcrCurrent.PCBase.Size != sizeof(PCRecRC))
					{
						sprintf_s(szMsg, MAX_MSG, MSG_IPA0105W, dwRecs + 1, pcrCurrent.PCBase.Type, pcrCurrent.PCBase.Size); 
						std::cout << szMsg << std::endl;
						dwPosLo = pcrCurrent.PCBase.Size - (sizeof(int) * 2);
#ifdef WIN32
						if(SetFilePointer(hfIPR, dwPosLo, (PLONG) &dwPosHi, FILE_CURRENT) == MAXWORD)
						{
							dwWinError = GetLastError();
							sprintf_s(szMsg, MAX_MSG, MSG_IPA0104E, dwPosLo);
							std::cout << szMsg << std::endl;
							GetWINAPIError(dwWinError, szMsg);
							std::cout << szMsg << std::endl;
							bKeepReading = FALSE;
						}
#endif
#ifdef UNIX
						offSrc = dwPosLo;
						offResult = lseek(hfIPR, offSrc, SEEK_CUR);
						if (offResult == -1)
						{
							sprintf_s(szMsg, MAX_MSG, MSG_IPA0104E, dwPosLo);
							std::cout << szMsg << std::endl;
							GetUNIXAPIError(errno, szMsg);
							std::cout << szMsg << std::endl;
							bKeepReading = FALSE;
						}
#endif
					}
					else
					{
						//  Read the remainder of the record
#ifdef WIN32
						if (!ReadFile(hfIPR, &pcrCurrent.PCBase.TID, pcrCurrent.PCBase.Size - (sizeof(int) * 2), &dwBytesRead, NULL))
						{						
							dwWinError = GetLastError();
							std::cout << MSG_IPA0106E << std::endl;
							GetWINAPIError(dwWinError, szMsg);
							std::cout << szMsg << std::endl;
							bKeepReading = FALSE;
						}
#endif
#ifdef UNIX
						stRead = read(hfIPR, &pcrCurrent.PCBase.TID, pcrCurrent.PCBase.Size - (sizeof(int) * 2));
						dwBytesRead = stRead;
						if (stRead == -1)
						{
							std::cout << MSG_IPA0106E << std::endl;
							GetUNIXAPIError(errno, szMsg);
							std::cout << szMsg << std::endl;
							bKeepReading = FALSE;
						}
#endif
						else
						{
							if (dwBytesRead != (pcrCurrent.PCBase.Size - (sizeof(int) * 2)))
							{
								//  EOF
								std::cout << MSG_IPA0102I << std::endl;
								bKeepReading = FALSE;
							}
							else
							{
								//  If this is the first event from a thread then set the time and clear the dwell
								if (pcrCurrent.PCBase.EventNo == 1)
								{
									pcrCurrent.PCBase.EventTime = pcrCurrent.Dwell;
									pcrCurrent.Dwell = 0;
								}	

								//  Perform the necessary output/analysis on the record
								if (iOutputs & OUTPUT_DUMP) DumpThisRecord(&pcrCurrent);
								if ((iOutputs & OUTPUT_STATS) || (iOutputs & OUTPUT_CHART))
								{
									//  Accumulate the dwell time and mill time in the appropriate frequency analysers
									asfaDwell->RecordDataPoint(pcrCurrent.Dwell);
									asfaMill->RecordDataPoint(pcrCurrent.Mill);
								}
							}
						}
					}
				}
			}
		}
		if (bKeepReading) dwRecs++;
	}

	//  If requested print the stats report
	if (iOutputs & OUTPUT_STATS) PrintStatsReport(asfaDwell, asfaMill);

	//  If requested print the stats charts
	if (iOutputs & OUTPUT_CHART) PrintCharts(asfaDwell, asfaMill);

	//  Show number of records processed
	sprintf_s(szMsg, MAX_MSG, MSG_IPA0101I, dwRecs);
	std::cout << szMsg << std::endl;

	//  Delete the analysers
	delete asfaDwell;
	delete asfaMill;

	return;
}

//  DumpThisRecord
//
//  This function will print the contents of the passed record
//
//  Parameters
//
//  1		-	PCRecRC *		-  Pointer to the record to be dumped
//
//  Returns
//
//  API Use Notes:
//
//
void DumpThisRecord(PCRecRC *pcrRec)
{
	char				szMsg[MAX_MSG + 1];				//  OS message buffer

	sprintf_s(szMsg, MAX_MSG, DUMPREC_FORMAT, pcrRec->PCBase.EventTime,
											  pcrRec->PCBase.TID,
											  pcrRec->PCBase.EventNo,
											  pcrRec->DPriority,
											  pcrRec->Dwell,
											  pcrRec->Mill);
	std::cout << szMsg << std::endl;
	return;
}

//  PrintStatsReport
//
//  This function will print the stats report of Dwell & Mill time
//
//  Parameters
//
//  1  -  AutoScaleFrequencyAnalyser *		-  The Analyser for Dwell time
//  2  -  AutoScaleFrequencyAnalyser *		-  The Analyser for Mill time
//
//  Returns
//
//  API Use Notes:
//
void PrintStatsReport(AutoScaleFrequencyAnalyser *asfaDwell, AutoScaleFrequencyAnalyser *asfaMill)
{
	BOOL				bPrintEntry;						//  Print control
	DWORD				dwCount;							//  Count
	DWORD				dwBucketLow;						//  Bucket low value
	DWORD				dwBucketHigh;						//  Bucket high value
	DWORD				dwModeCount;						//  Modal count
	DWORD				dwMedianCount;						//  Median count
	int					iIndex;								//  Generic index
	int					OutCount;							//  Outlier counts
	int					iLabelSize;							//  Size of data labels
	int					iDataSize;							//  Size of the data space
	unsigned long long	aSigma;								//  Accumulator of values
	DWORD				aCount;								//  Counter
	double				vMean;								//  Computed Mean
	double				aDeviation;							//  Accumulator of Deviations
	double				fWork1;								//  Work variable
	double				fWork2;								//  Work variable
	double				vStdDev;							//  Standard Deviation
	double				vMode;								//  Modal Value
	double				vMedian;							//  Median Value
	char				szMsg[MAX_MSG + 1];					//  Generic message buffer

	//  Clear the accumulators
	aSigma = 0;
	aCount = 0;
	vMean = 0.0;
	vStdDev = 0.0;
	aDeviation = 0.0;
	dwModeCount = 0;
	vMode = 0.0;
	dwMedianCount = 0;
	vMedian = 0.0;

	//  Report on the dwell times

	std::cout << std::endl << "Dwell Times." << std::endl << std::endl;
	bPrintEntry = FALSE;
	iDataSize = 100;

	//  Calculate the width of data labels to use
	dwCount = asfaDwell->GetFrequency(iDataSize - 1, &dwBucketLow, &dwBucketHigh);
	sprintf_s(szMsg, MAX_MSG, "%i", dwBucketHigh);
	iLabelSize = (int) strlen(szMsg);

	for (iIndex = iDataSize - 1; iIndex >= 0; iIndex--)
	{
		dwCount = asfaDwell->GetFrequency(iIndex, &dwBucketLow, &dwBucketHigh);
		if (dwCount > 0) bPrintEntry = TRUE;
		if (bPrintEntry)
		{
			sprintf_s(szMsg, MAX_MSG, STATREC_FORMAT, iLabelSize, dwBucketLow, iLabelSize, dwBucketHigh, dwCount);
			std::cout << szMsg << std::endl;
		}

		//  Accumulate the values
		aCount += dwCount;
		aSigma += ((dwBucketLow + dwBucketHigh) / 2) * dwCount;
	}

	std::cout << std::endl;

	//  Compute the mean
	fWork1 = (double) aSigma;
	fWork2 = (double) aCount;
	vMean = fWork1 / fWork2;

	//  Make a second pass over the data to compute the standard deviation
	for (iIndex = iDataSize - 1; iIndex >= 0; iIndex--)
	{
		dwCount = asfaDwell->GetFrequency(iIndex, &dwBucketLow, &dwBucketHigh);
		//  Accumulate the deviation
		if (dwCount > 0)
		{
			aSigma = ((dwBucketLow + dwBucketHigh) / 2);
			fWork1 = (double) aSigma;
			//  Check for mode change
			if (dwCount > dwModeCount)
			{
				dwModeCount = dwCount;
				vMode = fWork1;
			}
			//  Check for Median reached
			dwMedianCount += dwCount;
			if ((dwMedianCount >= (aCount / 2)) && (vMedian == 0.0))
			{
				vMedian = fWork1;
			}
			fWork1 = fWork1 - vMean;
			fWork1 = fWork1 * dwCount;
			aDeviation += fWork1 * fWork1;
		}
	}

	//  Compute the standard deviation
	fWork1 = (double) aCount;
	vStdDev = sqrt(aDeviation / fWork1);

	//  Print the mean and standard deviation
	sprintf_s(szMsg, MAX_MSG, MREC_FORMAT, aCount, vMean, vStdDev, vMode, dwModeCount, vMedian);
	std::cout << szMsg << std::endl;

	//  Report on the oultliers
	OutCount = asfaDwell->GetLowOutlierCount();
	if (OutCount > 0)
	{
		sprintf_s(szMsg, MAX_MSG," Low outliers: %i", asfaDwell->GetLowOutlier(0));
		for (iIndex = 1; iIndex <OutCount; iIndex++)
		{
			sprintf_s(szMsg, MAX_MSG, "%s, %i", szMsg, asfaDwell->GetLowOutlier(iIndex));
		}
		sprintf_s(szMsg, MAX_MSG, "%s.", szMsg);
		std::cout << szMsg << std::endl;
	}
	OutCount = asfaDwell->GetHighOutlierCount();
	if (OutCount > 0)
	{
		sprintf_s(szMsg, MAX_MSG," High outliers: %i", asfaDwell->GetHighOutlier(0));
		for (iIndex = 1; iIndex <OutCount; iIndex++)
		{
			sprintf_s(szMsg, MAX_MSG, "%s, %i", szMsg, asfaDwell->GetHighOutlier(iIndex));
		}
		sprintf_s(szMsg, MAX_MSG, "%s.", szMsg);
		std::cout << szMsg << std::endl;
	}
	std::cout << std::endl;

	//  Report on the mill times
	std::cout << std::endl << "Mill Times." << std::endl << std::endl;
	bPrintEntry = FALSE;

	//  Clear the accumulators
	aSigma = 0;
	aCount = 0;
	vMean = 0;
	aDeviation = 0;
	dwModeCount = 0;
	vMode = 0.0;
	dwMedianCount = 0;
	vMedian = 0.0;

	//  calculate the width of data labels to use
	dwCount = asfaMill->GetFrequency(iDataSize - 1, &dwBucketLow, &dwBucketHigh);
	sprintf_s(szMsg, MAX_MSG, "%i", dwBucketHigh);
	iLabelSize = (int) strlen(szMsg);

	for (iIndex = iDataSize - 1; iIndex >= 0; iIndex--)
	{
		dwCount = asfaMill->GetFrequency(iIndex, &dwBucketLow, &dwBucketHigh);
		if (dwCount > 0) bPrintEntry = TRUE;
		if (bPrintEntry)
		{
			sprintf_s(szMsg, MAX_MSG, STATREC_FORMAT, iLabelSize, dwBucketLow, iLabelSize, dwBucketHigh, dwCount);
			std::cout << szMsg << std::endl;
		}

		//  Accumulate the values
		aCount += dwCount;
		aSigma += ((dwBucketLow + dwBucketHigh) / 2) * dwCount;
	}

	std::cout << std::endl;

	//  Compute the mean
	fWork1 = (double) aSigma;
	fWork2 = (double) aCount;
	vMean = fWork1 / fWork2;

	//  Make a second pass over the data to compute the standard deviation
	for (iIndex = iDataSize - 1; iIndex >= 0; iIndex--)
	{
		dwCount = asfaMill->GetFrequency(iIndex, &dwBucketLow, &dwBucketHigh);
		//  Accumulate the deviation
		if (dwCount > 0)
		{
			aSigma = ((dwBucketLow + dwBucketHigh) / 2);
			fWork1 = (double) aSigma;
			//  Check for mode change
			if (dwCount > dwModeCount)
			{
				dwModeCount = dwCount;
				vMode = fWork1;
			}
			//  Check for Median reached
			dwMedianCount += dwCount;
			if ((dwMedianCount >= (aCount / 2)) && (vMedian == 0.0))
			{
				vMedian = fWork1;
			}
			fWork1 = fWork1 - vMean;
			fWork1 = fWork1 * dwCount;
			aDeviation += fWork1 * fWork1;
		}
	}

	//  Compute the standard deviation
	fWork1 = (double) aCount;
	vStdDev = sqrt(aDeviation / fWork1);

	//  Print the mean and standard deviation
	sprintf_s(szMsg, MAX_MSG, MREC_FORMAT, aCount, vMean, vStdDev, vMode, dwModeCount, vMedian);
	std::cout << szMsg << std::endl;

	//  Report on the oultliers
	OutCount = asfaMill->GetLowOutlierCount();
	if (OutCount > 0)
	{
		sprintf_s(szMsg, MAX_MSG," Low outliers: %i", asfaMill->GetLowOutlier(0));
		for (iIndex = 1; iIndex <OutCount; iIndex++)
		{
			sprintf_s(szMsg, MAX_MSG, "%s, %i", szMsg, asfaMill->GetLowOutlier(iIndex));
		}
		sprintf_s(szMsg, MAX_MSG, "%s.", szMsg);
		std::cout << szMsg << std::endl;
	}
	OutCount = asfaMill->GetHighOutlierCount();
	if (OutCount > 0)
	{
		sprintf_s(szMsg, MAX_MSG," High outliers: %i", asfaMill->GetHighOutlier(0));
		for (iIndex = 1; iIndex <OutCount; iIndex++)
		{
			sprintf_s(szMsg, MAX_MSG, "%s, %i", szMsg, asfaMill->GetHighOutlier(iIndex));
		}
		sprintf_s(szMsg, MAX_MSG, "%s.", szMsg);
		std::cout << szMsg << std::endl;
	}
	std::cout << std::endl;

	return;
}

#ifdef WIN32
//  GetWINAPIError
//
//  This function will convert a Windows API error code into a text message
//
//  Parameters
//
//  1  -  DWORD				-  The Windows API error code
//  2  -  char *			-  Pointer to the buffer where the error text will be written
//
//  Returns
//
//  API Use Notes:
//
void GetWINAPIError(DWORD dwErrorCode, char *szMsgBuffer)
{
	DWORD				dwMsgLen;						//  Length of message
	DWORD				dwWECode;						//  Alternate Error Code
	char				szMsg[MAX_MSG + 1];				//  OS message buffer

	//  This function will return a formatted windows API error message
	dwMsgLen = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, dwErrorCode, 0, szMsg, MAX_MSG, NULL);
	if (dwMsgLen == 0)
	{
		//  Error message could not be retrieved - try substituting the failure of formatting the message
		dwWECode = GetLastError();
		dwMsgLen = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, dwWECode, 0, szMsg, MAX_MSG, NULL);
	}
	sprintf_s(szMsgBuffer, MAX_MSG, "Platform API Error: 0x%04X - %s.", dwErrorCode, szMsg);
	return;
}
#endif

#ifdef UNIX
//  GetUNIXAPIError
//
//  This function will retirn the formatted error message for the latest API error.
//
//  Parameters
//	1 - int				-  The API Error Code
//  2 - char *			-  Pointer to a string buffer where the formatted error message will be returned
//
//  Returns
//
//  API Use Notes:
//
//
void GetUNIXAPIError(int iErrorCode, char *szMsgBuffer)
{
	//  Return the formatted error code
	sprintf_s(szMsgBuffer, MAX_MSG, "Platform API Error: 0x%04X - %s.", iErrorCode, strerror(iErrorCode));
	return;
}
#endif

//  PrintCharts
//
//  This function will print the histograms of the dwell and mill time
//
//  Parameters
//
//  1  -  AutoScaleFrequencyAnalyser *		-  The Analyser for Dwell time
//  2  -  AutoScaleFrequencyAnalyser *		-  The Analyser for Mill time
//
//  Returns
//
//  API Use Notes:
//
void PrintCharts(AutoScaleFrequencyAnalyser *asfaDwell, AutoScaleFrequencyAnalyser *asfaMill)
{
	//  Produce the histogram of the dwell time
	PrintHistogram("Dwell Times", asfaDwell);

	//  Produce the histogram of the mill time
	PrintHistogram("Mill Times", asfaMill);
	return;
}

//  PrintHistogram
//
//  This function will print a histograms of the passed data
//
//  Parameters
//
//  1  -  char *							-  Title for the chart
//  2  -  AutoScaleFrequencyAnalyser *		-  The Analyser for Mill time
//
//  Returns
//
//  API Use Notes:
//
void PrintHistogram(char *szTitle, AutoScaleFrequencyAnalyser *asfaData)
{
	AutoScaleFrequencyAnalyser		*asfaFreq;							//  Auto Scale array for frequecies
	int								iIndex;								//  Generic index
	int								jIndex;								//  Generic index
	DWORD							dwFreq;								//  Frequency value
	DWORD							dwCellMax;							//  Top of cell value
	DWORD							dwCellMin;							//  Bottom of cell value
	DWORD							dwHi;								//  High Bound
	DWORD							dwLo;								//  Low Bound
	int								iLabelSize;							//  Label size
	int								iDataSize;							//  Size of the data space
	int								iFreqSize;							//  Size of frequency space
	char							szSpacer[15];						//  Variable spacer
	char							szMsg[(MAX_MSG * 2) + 1];			//  Generic message buffer
	char							szDataLine[150];					//  Space to build a data line

	strcpy_s(szSpacer, 14, "          ");

	//  Build a new auto scale array to scale the frequencies
	asfaFreq = new AutoScaleFrequencyAnalyser(50, 10, 5);
	asfaFreq->DisallowOutliers();
	iDataSize = 100;
	iFreqSize = 50;

	//  Populate the array with the frequencies - this will collapse the data space into 50 points
	for (iIndex = (iDataSize - 1); iIndex >= 0; iIndex--)
	{
		dwFreq = asfaData->GetFrequency(iIndex, &dwCellMin, &dwCellMax);
		if (dwFreq > 0) asfaFreq->RecordDataPoint(dwFreq);
	}

	//  Build and write the title line
	std::cout << std::endl;
	sprintf_s(szMsg, MAX_MSG * 2, "   Frequency Analysis of %s.", szTitle);
	std::cout << szMsg << std::endl << std::endl;

	//  Determine the size to use for the frequency scale
	dwFreq = asfaFreq->GetFrequency(iFreqSize - 1, &dwCellMin, &dwCellMax);
	dwCellMax++;
	sprintf_s(szMsg, MAX_MSG * 2, "%i", dwCellMax);
	iLabelSize = (int) strlen(szMsg);
#ifdef _DEBUG
	printf("+DEBUG: Label Size has been computed as %i for '%i'.\r\n", iLabelSize, dwCellMax);
#endif

	//  Iterate over the frequency array
	for (iIndex = (iFreqSize - 1); iIndex >= 0; iIndex--)
	{
		dwFreq = asfaFreq->GetFrequency(iIndex, &dwCellMin, &dwCellMax);

		//  Build the histogram line for this data interval
		for (jIndex = 0; jIndex < iDataSize; jIndex++)
		{
			dwFreq = asfaData->GetFrequency(jIndex, &dwLo, &dwHi);
			if (dwFreq > dwCellMin) szDataLine[jIndex] = '*';
			else szDataLine[jIndex] = ' ';
		}
		szDataLine[iDataSize] = '\0';

		// Format the completed print line
		if ((iIndex + 1) % 5 == 0)
		{
			//  Print line with Y axis label
			sprintf_s(szMsg, MAX_MSG * 2, HISTLLINE_FORMAT, iLabelSize, dwCellMax + 1, szDataLine);
		}
		else
		{
			//  Naked print line
			szSpacer[iLabelSize] = '\0';
			sprintf_s(szMsg, MAX_MSG * 2, HISTNLINE_FORMAT, szSpacer, szDataLine);
			szSpacer[iLabelSize] = ' ';
		}
		std::cout << szMsg << std::endl;
	}

	//  Write the x-axis line
	szSpacer[iLabelSize] = '\0';
	sprintf_s(szMsg, MAX_MSG * 2, HISTXAXIS_FORMAT, szSpacer, "---------|");
	for (jIndex = 0; jIndex < iDataSize; jIndex++)
	{
		if (jIndex % 10 == 0) sprintf_s(szMsg, MAX_MSG * 2, "%s%s", szMsg, "---------|");
	}
	szSpacer[iLabelSize] = ' ';
	std::cout << szMsg << std::endl;

	//  Write the x-axis label
	szSpacer[iLabelSize] = '\0';
	sprintf_s(szMsg, MAX_MSG * 2, "  %s   ", szSpacer);
	szSpacer[iLabelSize] = ' ';
	
	dwFreq = asfaData->GetFrequency(iDataSize - 1, &dwCellMin, &dwCellMax); 
	sprintf_s(szDataLine, iDataSize, "%i", dwCellMax);
	iLabelSize = (int) strlen(szDataLine);
	for (jIndex = 0; jIndex < iDataSize; jIndex++)
	{
		dwFreq = asfaData->GetFrequency(jIndex, &dwLo, &dwHi);
		if (jIndex % 10 == 0)
		{
			szSpacer[10 - iLabelSize] = '\0';
			sprintf_s(szMsg, MAX_MSG * 2, "%s%.*i%s", szMsg, iLabelSize, dwLo, szSpacer);
			szSpacer[10 - iLabelSize] = ' ';
		}
	}
	szSpacer[10 - iLabelSize] = '\0';
	sprintf_s(szMsg, MAX_MSG * 2, "%s%.*i%s", szMsg, iLabelSize, dwHi + 1, szSpacer);
	szSpacer[10 - iLabelSize] = ' ';
	std::cout << szMsg << std::endl << std::endl;
#ifdef _DEBUG
	printf("+DEBUG: Label Size was computed as %i for '%i'.\r\n", iLabelSize, dwCellMax);
#endif
	//  Delete the frequency analyser
	delete asfaFreq;
	return;
}

